﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using TMPro;
using UnityEngine;
using UnityEngine.UI;

//MessageBoxManager is a singleton class responsible for managing the MessageBox UI and providing an API for it
public class MessageBoxManager : MonoBehaviour
{
    private enum MessageBoxButtons
    {
        Left,
        Centre,
        Right
    }

    //UI elements
    private GameObject _ParentContainerObject;
    private GameObject _MessageBoxObject;
    private TMP_Text _TitleText;
    private TMP_Text _MessageText;
    private List<string> _Buttons;
    private Button _LeftButton;
    private Button _CentreButton;
    private Button _RightButton;

    public int SelectedButton { get; private set; }
    public bool IsShowingMessageBox { get; private set; }

    private TaskCompletionSource<bool> ClosedTask;  //Task to allow waiting of MessageBox to close

    //Singleton
    private static MessageBoxManager _Instance;

    public static MessageBoxManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<MessageBoxManager>();
            }

            return _Instance;
        }
    }

    /// <summary>
    /// Sets up the message box's internal UI state from the base parent messageBoxObject
    /// </summary>
    /// <param name="messageBoxObject">The base message box object</param>
    /// <param name="parentContainerObject">The parent container of the message box object</param>
    public void SetMessageBoxObject(GameObject messageBoxObject, GameObject parentContainerObject = null)
    {
        //Set up the state
        _ParentContainerObject = parentContainerObject;
        _MessageBoxObject = messageBoxObject;
        ClosedTask = new TaskCompletionSource<bool>();
        SelectedButton = -1;
        _Buttons = null;

        //Set up and access the UI components
        _TitleText = _MessageBoxObject.FindChild("BackgroundPanelOrange").FindChild("TitleText").GetComponent<TMP_Text>();
        _MessageText = _MessageBoxObject.FindChild("BackgroundPanelGrey").FindChild("MessageText").GetComponent<TMP_Text>();
        _LeftButton = _MessageBoxObject.FindChild("BackgroundPanelGrey").FindChild("Buttons").FindChild("LeftButton").GetComponent<Button>();
        _CentreButton = _MessageBoxObject.FindChild("BackgroundPanelGrey").FindChild("Buttons").FindChild("CentreButton").GetComponent<Button>();
        _RightButton = _MessageBoxObject.FindChild("BackgroundPanelGrey").FindChild("Buttons").FindChild("RightButton").GetComponent<Button>();
        _MessageBoxObject.SetActive(false); //We've definitely got the UI controls so now it's known to be a MessageBox and safe to disable

        _LeftButton.onClick.AddListener(delegate { OnButtonSelected(MessageBoxButtons.Left); });
        _CentreButton.onClick.AddListener(delegate { OnButtonSelected(MessageBoxButtons.Centre); });
        _RightButton.onClick.AddListener(delegate { OnButtonSelected(MessageBoxButtons.Right); });
    }

    /// <summary>
    /// Sets up the buttons of the message box based on the amount and text
    /// </summary>
    private void SetupButtons()
    {
        //Setup the buttons and the navigations then show
        if (_Buttons.Count == 1)
        {
            _LeftButton.gameObject.SetActive(false);
            _RightButton.gameObject.SetActive(false);

            _CentreButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(_Buttons[0]);
            _CentreButton.gameObject.SetActive(true);

            _CentreButton.OnSelect(null);
        }

        else if (_Buttons.Count == 2)
        {
            _CentreButton.gameObject.SetActive(false);

            _LeftButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(_Buttons[0]);
            _RightButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(_Buttons[1]);
            _LeftButton.gameObject.SetActive(true);
            _RightButton.gameObject.SetActive(true);

            Navigation leftButtonNavigation = _LeftButton.navigation;
            leftButtonNavigation.mode = Navigation.Mode.Explicit;
            leftButtonNavigation.selectOnRight = _RightButton;
            _LeftButton.navigation = leftButtonNavigation;

            Navigation rightButtonNavigation = _RightButton.navigation;
            rightButtonNavigation.mode = Navigation.Mode.Explicit;
            rightButtonNavigation.selectOnLeft = _LeftButton;
            _RightButton.navigation = rightButtonNavigation;
        }

        else
        {
            _LeftButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(_Buttons[0]);
            _CentreButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(_Buttons[1]);
            _RightButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(_Buttons[2]);
            _LeftButton.gameObject.SetActive(true);
            _CentreButton.gameObject.SetActive(true);
            _RightButton.gameObject.SetActive(true);

            Navigation leftButtonNavigation = _LeftButton.navigation;
            leftButtonNavigation.mode = Navigation.Mode.Explicit;
            leftButtonNavigation.selectOnRight = _CentreButton;
            _LeftButton.navigation = leftButtonNavigation;

            Navigation rightButtonNavigation = _RightButton.navigation;
            rightButtonNavigation.mode = Navigation.Mode.Explicit;
            rightButtonNavigation.selectOnLeft = _CentreButton;
            _RightButton.navigation = rightButtonNavigation;

            Navigation centreButtonNavigation = _CentreButton.navigation;
            centreButtonNavigation.mode = Navigation.Mode.Explicit;
            centreButtonNavigation.selectOnLeft = _LeftButton;
            centreButtonNavigation.selectOnRight = _RightButton;
            _CentreButton.navigation = centreButtonNavigation;
        }
    }

    /// <summary>
    /// Tries to show the message box, setting up its state and returning the selected button index if applicable
    /// </summary>
    /// <returns>The index of the selected button</returns>
    private async Task<int> TryShowMessageBox()
    {
        if (_Buttons.Count > 0)
        {
            ClosedTask = new TaskCompletionSource<bool>();
            SelectedButton = -1;

            _MessageBoxObject.SetActive(true);
            SetParentActive(true);
            IsShowingMessageBox = true;

            SetupButtons();

            await ClosedTask.Task;

            //The MessageBox has closed, hide it, show the page again and return the index of the button selected
            _MessageBoxObject.SetActive(false);
            SetParentActive(false);
            IsShowingMessageBox = false;
            return SelectedButton;
        }

        return -1;
    }

    /// <summary>
    /// Shows the message box, returning the index of the selected button
    /// </summary>
    /// <param name="title">The title of the message box</param>
    /// <param name="text">The text to show</param>
    /// <returns></returns>
    public async Task<int> Show(string title, string text)
    {
        //Setup the text and buttons
        _TitleText.SetText(title);
        _MessageText.SetText(text);
        _Buttons = new List<string>() { "OK" };

        return await TryShowMessageBox();
    }

    /// <summary>
    /// Shows the message box, returning the index of the selected button
    /// </summary>
    /// <param name="title">The title of the message box</param>
    /// <param name="text">The text to show</param>
    /// <param name="buttons">A list of strings for the text for each button</param>
    /// <returns></returns>
    public async Task<int> Show(string title, string text, List<string> buttons)
    {
        //Setup the text and buttons
        _TitleText.SetText(title);
        _MessageText.SetText(text);
        _Buttons = buttons;

        return await TryShowMessageBox();
    }

    /// <summary>
    /// Handler for when a button is selected, converts the logically positioned button into the correct index based on how many buttons are present
    /// </summary>
    /// <param name="selectedButton">The logically positioned selected button</param>
    private void OnButtonSelected(MessageBoxButtons selectedButton)
    {
        //We selected a button, set the index as appropriate for the buttons visible
        if (selectedButton == MessageBoxButtons.Left)
        {
            SelectedButton = 0;
        }

        else if (selectedButton == MessageBoxButtons.Centre)
        {
            if (_Buttons.Count == 1)
            {
                SelectedButton = 0;
            }

            else
            {
                SelectedButton = 1;
            }
        }

        else if (selectedButton == MessageBoxButtons.Right)
        {
            if (_Buttons.Count == 2)
            {
                SelectedButton = 1;
            }

            else
            {
                SelectedButton = 2;
            }
        }

        ClosedTask.SetResult(true); //Button selected, time to close
    }

    /// <summary>
    /// Sets the active state of the parent container object
    /// </summary>
    /// <param name="active">Should the parent be set as active?</param>
    private void SetParentActive(bool active)
    {
        if(_ParentContainerObject != null)
        {
            _ParentContainerObject.SetActive(active);
        }
    }
}
